%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTICLE DYNAMICS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1, '\n === START: particle dynamics ====\n\n');

clear all;
close all hidden;

inputdir  = 'd_geom/setup_001/';
outputdir = 'd_sim/setup_001_e10L/';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% re-import geometry
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1, 'Import geometry \n');
loadfile = [inputdir, 'obstacles.mat'];
load(loadfile, 'row_obs');


[opoints,odim,obs_per_row,rows] = size(row_obs);

loadfile = [inputdir, 'boundary.mat'];
load(loadfile, 'corn');

[corners,cdim]= size(corn);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% re-plot
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close(figure(1))
figure(1)

line(corn(:,1), corn(:,2))


for r=1:rows
    for i=1:obs_per_row
        line(row_obs(:,1,i,r), row_obs(:,2,i,r))
    end
end

%axis xy;
%axis tight;
%axis image
%axis equal;
hold on;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% extract segments
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Lx = corn(2,1);
Ly = corn(3,2);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Dynamics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% particle seeding
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1, 'Generate initial conditions \n');

N=2000;

V=10;  %chlamy speed = 100 mu/sec; 
% this means that unit time step T=d/V= 1 sec
mu = 5; %mean run time in sec

Tmax = 10 * 60; % 10 minutes
Tmax = 6000;

output_max  = 10;
output_dt   = Tmax / output_max; 

elasticity = 1; % choose > 0, 1= elastic collision
MyMarkerSize=1;

% prevent particle seedings within obstacles 

boxes    = rows + 1; 
box1_x   = Lx /(2*boxes); % x-center of first box
box_wid   = 2* box1_x;      


max_x_row = zeros(rows,1);
min_x_row = zeros(rows,1);
wid_x_row = zeros(rows,1);

for r=1:rows
   max_x_row(r) = max(row_obs(:,1,1,r));
   min_x_row(r) = min(row_obs(:,1,1,r));
   wid_x_row(r) = max_x_row(r) - min_x_row(r);
end

% reduced box width for initial seeding

obs_wid     = max(wid_x_row(:));
red_box_wid = box_wid - obs_wid; 

% random initial positions 

pos = zeros(N, 2);

% random box assignment
rb_shift =  ( random('unid', boxes, N, 1) - ones(N,1) ) * box_wid;

x_pos = box1_x + rb_shift + random('unif', -0.48 * red_box_wid, 0.48 * red_box_wid, N, 1 );
y_pos = random('Uniform', 0, Ly, N, 1 );

pos(:,1) = x_pos;
pos(:,2) = y_pos;

% test IC
%pos(:,1) = 5 ;
%pos(:,2) = 7;

% show particles in plot


plot(pos(:,1), pos(:,2), 'rs', 'Marker', 'o', 'MarkerSize', MyMarkerSize);
axis image;
xlabel('X (bodylength)');
ylabel('Y (bodylength)');


% random initial velocities

vel = zeros(N,2);

v_phi=random('unif', 0, 2*pi, N, 1 );
vel(:,1) = V * cos(v_phi);
vel(:,2) = V * sin(v_phi);

% test IC
%v_phi= pi;
%vel(:,1) = V * cos(v_phi);
%vel(:,2) = V * sin(v_phi);



figure(2);
scatterhist(vel(:,1), vel(:,2));
xlabel('V_x (bodylength / sec)');
ylabel('V_y (bodylength / sec)');

close(figure(3));
figure(3);
hist(x_pos,boxes);
xlabel('X (bodylength)');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compute collision table for init cond
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

time = 0;
coll_count = 0;
tumb_count = 0;
bound_count = 0;
obs_count = 0;


coll_dt    = 1e10 * ones(N,1);
coll_para = zeros(N,8);
coll_type = zeros(N); % 0= tumble, 1= wall, 2=obstacle

tc_crit =0.00001; % needed to avoid entering into obstacles

for n = 1:N
    
    %fprintf(1, '\nPARTICLE = %d\n', n);
         
    x  = pos(n,1);
    y  = pos(n,2);
    vx = vel(n,1);
    vy = vel(n,2);
    
    
    % boundary collisions
    for c = 1:(corners-1)
        %fprintf(1, '\nboundary=%d\t', c);
        
        bx = corn(c,1);
        by = corn(c,2);
        cx = corn(c+1,1);
        cy = corn(c+1,2);
        
        tc=f_coll_time(x, y, vx, vy, bx, by, cx, cy);
        
        if tc < coll_dt(n) && tc > tc_crit
            coll_dt(n) = tc;
            coll_type(n)= 1;
            coll_para(n,:) =[x, y, vx, vy, bx, by, cx, cy];       
        end
        
    end
    
    % obstacle collisions
    
    for r = 1:rows
        for o = 1:obs_per_row
            
            %fprintf(1, '\n obstacle=(%d, %d)\n', r, o);
            
            for c = 1:(opoints-1)
                
                bx = row_obs(c,1,o,r);
                by = row_obs(c,2,o,r);
                cx = row_obs(c+1,1,o,r);
                cy = row_obs(c+1,2,o,r);
                
                tc=f_coll_time(x, y, vx, vy, bx, by, cx, cy);
                
                if tc < coll_dt(n) && tc > tc_crit
                    coll_dt(n) = tc;
                    coll_type(n)= 2;
                    coll_para(n,:) =[x, y, vx, vy, bx, by, cx, cy];
                end
                
            end
            
        end
    end
    
    % tumbling treated as independent collision
    
    bx = 0;
    by = 0;
    cx = 0;
    cy = 0;
    
   
    tc = exprnd(mu);
    %tc = Inf;
    if tc < coll_dt(n) && tc > tc_crit
        coll_dt(n) = tc;
        coll_type(n)= 0;
        coll_para(n,:) =[x, y, vx, vy, bx, by, cx, cy];
    end
    
end
    
%sort collision table by collision times

coll_tab = zeros(N, 12);
s_coll_tab = zeros(N, 12);

for n = 1:N
    coll_tab(n,:) = cat(2, n, time, coll_dt(n), coll_type(n), coll_para(n,:));
end

s_coll_tab = sortrows(coll_tab, 3);

%s_coll_tab(:,1:3)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% prepare output
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sys_evol = zeros(N, 6, output_max + 1);

output      = 1;

% save initial positions

for n = 1:N
    sys_evol(n,:,output)=[coll_tab(n,1), time, coll_tab(n,5), coll_tab(n,6), coll_tab(n,7), coll_tab(n,8) ];
end

output_time = output_dt;
   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% evolve system 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(1, 'Compute dynamics \n');

 
while time < Tmax
   
    time_shift =  s_coll_tab(1,3); 
    time = time + time_shift;
    
    % total time
    s_coll_tab(:,2) = s_coll_tab(:,2) + time_shift;
    
    % time to next collision
    s_coll_tab(:,3) = s_coll_tab(:,3) - time_shift;
    
    % new particle positions
    s_coll_tab(:,5) = s_coll_tab(:,5) + time_shift * s_coll_tab(:,7);
    s_coll_tab(:,6) = s_coll_tab(:,6) + time_shift * s_coll_tab(:,8);
    
  
    % correct for "escape" due to numerical error
    for n=1:N
         s_coll_tab(n,5) = min(max(0, s_coll_tab(n,5)) , Lx);
         s_coll_tab(n,6) = min(max(0, s_coll_tab(n,6)) , Ly);
         
         
         if s_coll_tab(n,5)> Lx
             fprintf(1, '\n error: x > L_x\n');
         end
         
         if s_coll_tab(n,5)< 0
             fprintf(1, '\n error: x < 0\n');
         end
         
         if s_coll_tab(n,6)> Ly
             fprintf(1, '\n error:  y > L_y\n');
         end
         
         if s_coll_tab(n,6) < 0
             fprintf(1, '\n error : y < 0 \n');
         end
    
         
         
          if s_coll_tab(n,3) < 0
             fprintf(1, '\n error: dt <  0\n');
         end
    end
    
      
   
   %s_coll_tab(:,1:3)

    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % perform collision
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    % tumble
    
    if  int8(s_coll_tab(1,4)) == 0
        
      phi_tumble = random('unif', 0, 2*pi, 1, 1 );
      vx_new = V * cos(phi_tumble);
      vy_new = V * sin(phi_tumble);
    
      tumb_count = tumb_count +1;
    end

    % boundary or obstacle collision
    
    refl = zeros(2,2);
    v_new = zeros(2,1);
    
    if  int8(s_coll_tab(1,4)) > 0
        
         vx = s_coll_tab(1,7);
         vy = s_coll_tab(1,8);
         
         bx = s_coll_tab(1, 9);
         by = s_coll_tab(1,10);
         cx = s_coll_tab(1,11);
         cy = s_coll_tab(1,12);
        
         hwx = (bx-cx)/norm([bx-cx, by-cy]);
         hwy = (by-cy)/norm([bx-cx, by-cy]);
         
         refl = eye(2) - transpose([hwx  hwy])* ([hwx  hwy]);
         
         v_new = transpose([vx, vy]) - (1+elasticity)* refl * transpose([vx, vy]);
         
         v_stretch =V/norm(v_new);
         
         vx_new = v_new(1,1) *  v_stretch;
         vy_new = v_new(2,1) *  v_stretch;
         
          if  int8(s_coll_tab(1,4)) == 1
               bound_count = bound_count + 1 ;
          end
         
          if  int8(s_coll_tab(1,4)) == 2
              obs_count = obs_count + 1 ;
          end
    end
    
    coll_count = coll_count + 1;
      
    % new particle velocities
    if abs(vx_new)> 0.001*V
        s_coll_tab(1,7) = vx_new;
    else
        s_coll_tab(1,7)=0;
        
    end
    
    if abs(vy_new)> 0.001*V
        s_coll_tab(1,8) = vy_new;
    else
        s_coll_tab(1,8)=0;
    end
    %fprintf(1, '\n test\n');
     
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % compute next collision
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   
    
    coll_dt_next = 1e6;
    
    x  =  s_coll_tab(1,5);
    y  =  s_coll_tab(1,6);
    vx =  s_coll_tab(1,7);
    vy =  s_coll_tab(1,8);
    
    
    % boundary collisions
    for c = 1:(corners-1)
        %fprintf(1, '\nboundary=%d\t', c);
        
        bx = corn(c,1);
        by = corn(c,2);
        cx = corn(c+1,1);
        cy = corn(c+1,2);
        
        tc=f_coll_time(x, y, vx, vy, bx, by, cx, cy);
        
        if tc <  coll_dt_next && tc > tc_crit;
            coll_dt_next   = tc;
            coll_type_next = 1;
            coll_para_next =[x, y, vx, vy, bx, by, cx, cy];       
        end
        
    end
    
    % obstacle collisions
    
    for r = 1:rows
        for o = 1:obs_per_row
            
            %fprintf(1, '\n obstacle=(%d, %d)\n', r, o);
            
            for c = 1:(opoints-1)
                
                bx = row_obs(c,1,o,r);
                by = row_obs(c,2,o,r);
                cx = row_obs(c+1,1,o,r);
                cy = row_obs(c+1,2,o,r);
                
                tc=f_coll_time(x, y, vx, vy, bx, by, cx, cy);
                
                if tc <  coll_dt_next && tc > tc_crit;
                    coll_dt_next   = tc;
                    coll_type_next = 2;
                    coll_para_next =[x, y, vx, vy, bx, by, cx, cy];
                end
                
            end
            
        end
    end
    
    % tumbling treated as independent collision
    
    bx = 0;
    by = 0;
    cx = 0;
    cy = 0;
    
    
    tc = exprnd(mu);
    %tc = Inf + 1;
        
    if tc <  coll_dt_next && tc > tc_crit;
        coll_dt_next   = tc;
        coll_type_next = 0;
        coll_para_next =[x, y, vx, vy, bx, by, cx, cy];
    end
    
    
    s_coll_tab(1,:) = cat(2, s_coll_tab(1,1), time, coll_dt_next, coll_type_next, coll_para_next);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % update & resort collision table
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if N >2 && s_coll_tab(1,3) > s_coll_tab(2,3) 
        s_coll_tab = sortrows(s_coll_tab, 3);
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % output
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    
    if time > output_time
        output = output +1;
        coll_tab = sortrows(s_coll_tab, 1);
        
        for n = 1:N
            sys_evol(n,:, output)=[coll_tab(n,1), time, coll_tab(n,5), coll_tab(n,6), coll_tab(n,7), coll_tab(n,8) ];
        end
        
        output_time = output_time + output_dt;
        
        fprintf(1, '\n');
        fprintf(1, '\t t =%4.2g', s_coll_tab(1,2));
        fprintf(1, '\t # col =%4.2g', coll_count);
        fprintf(1, '\t # obs =%4.2g', obs_count);
        fprintf(1, '\t # tum =%4.2g', tumb_count);
    end
    
%     fprintf(1, '\n');
%     fprintf(1, '%d', int8(s_coll_tab(1,4))); % type
%     fprintf(1, '\t t =%4.2g', s_coll_tab(1,2));      
%     fprintf(1, ' dt =%2.2g',   s_coll_tab(1,3));
%     fprintf(1, '\t x =%4.2g',   s_coll_tab(1,5));
%     fprintf(1, '\t y =%4.2g',   s_coll_tab(1,6));
%     fprintf(1, '\t vx =%4.2g',   s_coll_tab(1,7));
%     fprintf(1, '\t vy =%4.2g',   s_coll_tab(1,8));
%      fprintf(1, '\t vy =%4.2g', ...
%          norm([s_coll_tab(1,7), s_coll_tab(1,8)]));
     
    
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXPORT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


savefile = [outputdir, 'sys_evol.mat'];
save(savefile, 'sys_evol');



% show particles in plot
figure(1)
hold on;
for i=output:output
    plot(sys_evol(:,3,i), sys_evol(:,4,i), 'gs', 'Marker', 'o', 'MarkerSize', MyMarkerSize);
end

% for i=1:(output-1)
%     quiver(sys_evol(:,3,i), sys_evol(:,4,i),...
%         sys_evol(:,3,i+1)- sys_evol(:,3,i), ...
%         sys_evol(:,4,i+1)- sys_evol(:,4,i), 'r');
% end

axis image;
xlabel('X (bodylength)');
ylabel('Y (bodylength)');

%[coll_count, tumb_count, bound_count, obs_count];

fprintf(1, '\n=== done ===\n\n');

m_plot;

%%%
% later: tumble

%%%%%
% d    = 10 mu
% Ly   =  2 mm
% Lx   =  10 mm / 4 obstacle rows
% fill = 0.01
% N    =2000
% 10 min